/** @module types/error */


export class AggHouseError {
    /**
     * Class containing AggHouse error info.
     * @class AggHouseError
     * @memberof module:types/error
     * @param {string} message Error message.
     * @param {boolean} isRecoverable If the error can be recovered.
     */
    constructor(message, isRecoverable) {
        this.message_ = message;
        this.isRecoverable_ = isRecoverable;
    }

    /**
     * Get error name.
     * @return {string} Error name.
     */
    get name() {
        return 'AggHouseError';
    }

    /**
     * Get error message.
     * @return {string} Error message.
     */
    get message() {
        return this.message_;
    }

    /**
     * Get if error is recoverable.
     * @return {boolean} Error recoverbility.
     */
    get recoverable() {
        return this.isRecoverable_;
    }

    /**
     * Get error string.
     * @return {string} Error string.
     * @override
     */
    toString() {
        return (this.recoverable ? '[ ' : '[Fatal][') +
            this.name + '] ' + this.message;
    }
}

export class AggHouseInvalidArgumentError extends AggHouseError {
    /**
     * Class containing error info triggered by invalid arguments.
     * @class AggHouseInvalidArgumentError
     * @extends AggHouseError
     * @memberof module:types/error
     * @param {string} message Error message.
     */
    constructor(message) {
        super(message, true);
    }

    /**
     * Get error message.
     * @return {string} Error message.
     * @override
     */
    get name() {
        return 'AggHouseInvalidArgumentError';
    }
}

export class AggHouseRoomError extends AggHouseError {
    /**
     * Class containing error info triggered by room error.
     * @class AggHouseRoomError
     * @extends AggHouseError
     * @memberof module:types/error
     * @param {string} message Error message.
     */
    constructor(message) {
        super(message, true);
    }

    /**
     * Get error message.
     * @return {string} Error message.
     * @override
     */
    get name() {
        return 'AggHouseRoomError';
    }
}

export class AggHouseUnavailableError extends AggHouseError {
    /**
     * Class containing error info of unavailable operations.
     * @class AggHouseUnavailableError
     * @extends AggHouseError
     * @memberof module:types/error
     * @param {string} message Error message.
     */
    constructor(message) {
        super(message, false);
    }

    /**
     * Get error message.
     * @return {string} Error message.
     * @override
     */
    get name() {
        return 'AggHouseUnavailableError';
    }
}
